//=============================================================================

/*:
 * @plugindesc アクターが残り1人になるとBGMを変更します。
 * 
 * @author LefTonbo
 *
 * @param Crisis BGM
 * @type file
 * @dir audio/bgm/
 * @require 1
 * @desc ピンチ時に使用するBGM
 * @default battle2
 *
 * @param Crisis BGM Volume
 * @type number
 * @desc ピンチ時BGMの音量
 * @default 90
 *
 * @param Crisis BGM Pitch
 * @type number
 * @desc ピンチ時BGMのピッチ
 * @default 100
 *
 * @param One Actor Exclude
 * @type select
 * @option 変更しない
 * @value 0
 * @option 残りHPで判定
 * @value 1
 * @option そのまま
 * @value 2
 * @desc パーティが1人の時に適用するかどうか
 * @default 1
 *
 * @param One Actor Crisis HP
 * @type number
 * @decimals 2
 * @desc パーティが1人の時でHp判定するとき、何％以下でピンチとするか
 * @default 0.34
 *
 * @param Change Crisis Border
 * @type boolean
 * @on はい
 * @off いいえ
 * @desc 上記設定をUIに反映させるか
 * @default true
 *
 * @param Crisis Start State
 * @type number
 * @desc ピンチ状態になった瞬間、指定したステートをに加える
 * @default 3
 *
 * @param Crisis Turn State
 * @type number
 * @desc ピンチ状態の時、指定したステートを毎ターン開始時に加える
 * @default 3
 *
 * @help
 * アクターが残り1人になるとBGMを変更します。
 * またピンチ時に持続ステートを発生させます。
 */

//=============================================================================

var Imported = Imported || {};
Imported.LP_BattleCrisisBGM = true;

var LeftPlugin = LeftPlugin || {};

LeftPlugin.Parameters = PluginManager.parameters('LP_BattleCrisisBGM');
LeftPlugin.Param = LeftPlugin.Param || {};

LeftPlugin.Param.CrisisBGMName = String(LeftPlugin.Parameters['Crisis BGM']);
LeftPlugin.Param.CrisisBGMVolume = Number(LeftPlugin.Parameters['Crisis BGM Volume']);
LeftPlugin.Param.CrisisBGMPitch = Number(LeftPlugin.Parameters['Crisis BGM Pitch']);
LeftPlugin.Param.OneActorExclude = Number(LeftPlugin.Parameters['One Actor Exclude']);
LeftPlugin.Param.OneActorCrisisHp = Number(LeftPlugin.Parameters['One Actor Crisis HP']);
LeftPlugin.Param.ChangeCrisisColorBorder = Boolean(LeftPlugin.Parameters['Change Crisis Border']);
LeftPlugin.Param.CrisisStartState = Number(LeftPlugin.Parameters['Crisis Start State']);
LeftPlugin.Param.CrisisTurnState = Number(LeftPlugin.Parameters['Crisis Turn State']);

//=============================================================================

(function() {

//=============================================================================
// BattleManager
//=============================================================================

BattleManager.prevCrisisBGM = null;
BattleManager.isCrisis = false;

var _BattleManager_startBattle = BattleManager.startBattle;
BattleManager.startBattle = function(troopId, canEscape, canLose) {
  _BattleManager_startBattle.call(this, troopId, canEscape, canLose);
  this.prevCrisisBGM = null;
  this.isCrisis = false;
  this.checkPartyCrisis();
};

var _BattleManager_endTurn = BattleManager.endTurn;
BattleManager.endTurn = function() {
  _BattleManager_endTurn.call(this);
  if (this.isCrisis == false) {
    if (LeftPlugin.Param.CrisisStartState > 0 || LeftPlugin.Param.CrisisTurnState > 0) {
      var party = $gameParty.members();
      for (var i = 0; i < party.length; i++) {
        var member = party[i];
        member.removeState(LeftPlugin.Param.CrisisStartState);
        member.removeState(LeftPlugin.Param.CrisisTurnState);
      }
    }
  } else {
    if (LeftPlugin.Param.CrisisStartState > 0 || LeftPlugin.Param.CrisisTurnState > 0) {
      var party = $gameParty.members();
      for (var i = 0; i < party.length; i++) {
        var member = party[i];
        member.removeState(LeftPlugin.Param.CrisisStartState);
        if (member.hp <= 0) continue;
        member.addState(LeftPlugin.Param.CrisisTurnState);
      }
    }
  }
};

BattleManager.checkPartyCrisis = function() {
  var party = $gameParty.members();
  if (LeftPlugin.Param.OnePartyExclude == 0 && party.length <= 2) {
    return;
  }
  var lives = party.length;
  for (var i = 0; i < party.length; i++) {
    var member = party[i];
    if (LeftPlugin.Param.OneActorExclude == 1 && party.length == 1) {
      if (member.hp > LeftPlugin.Param.OneActorCrisisHp * member.mhp) {
        lives ++;
      }
    } else {
      if (member.hp <= 0) lives --;
    }
  }
  if (lives <= 1) {
    if (this.isCrisis == false) {
      if (LeftPlugin.Param.CrisisStartState > 0) {
        for (var i = 0; i < party.length; i++) {
          var member = party[i];
          if (member.hp <= 0) continue;
          member.addState(LeftPlugin.Param.CrisisStartState);
        }
      }
    }
    this.isCrisis = true;
    this.playCrisisBGM ();
  } else {
    this.isCrisis = false;
    this.stopCrisisBGM ();
  }
};

BattleManager.playCrisisBGM = function() {
  if (this.prevCrisisBGM !== null) return;
  this.prevCrisisBGM = AudioManager.saveBgm();
  var bgm = {
    name: LeftPlugin.Param.CrisisBGMName,
    volume: LeftPlugin.Param.CrisisBGMVolume,
    pitch: LeftPlugin.Param.CrisisBGMPitch,
    pan: 0
  };
  AudioManager.playBgm(bgm);
}

BattleManager.stopCrisisBGM = function() {
  if (this.prevCrisisBGM === null) return;
  AudioManager.playBgm(this.prevCrisisBGM, this.prevCrisisBGM.pos);
  AudioManager.fadeInBgm(0.5);
  this.prevCrisisBGM = null;
}

//================

if (Imported.YEP_VictoryAftermath) {

var _BattleManager_processNormalVictory = BattleManager.processNormalVictory;
BattleManager.processNormalVictory = function() {
  if ($gameSystem.skipVictoryMusic()) {
    this.stopCrisisBGM ();
  }
  _BattleManager_processNormalVictory.call(this);
}

}

//=============================================================================
// Game_BattlerBase
//=============================================================================

var _Game_BattlerBase_setHp = Game_BattlerBase.prototype.setHp;
Game_BattlerBase.prototype.setHp = function(hp) {
  _Game_BattlerBase_setHp.call(this, hp);
  if (this.isActor() && $gameParty.inBattle()) {
    BattleManager.checkPartyCrisis();
  }
};

if (LeftPlugin.Param.ChangeCrisisColorBorder) {
Game_BattlerBase.prototype.isDying = function() {
    return this.isAlive() && this._hp < this.mhp * LeftPlugin.Param.OneActorCrisisHp;
};
}

//=============================================================================

})();

// EOF
